/** @file   bullettable.h
 * @brief   Declaration of BulletTable - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:41 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_BULLETTABLE_H
#define H_WWW_BULLETTABLE_H

#include "bullet.h"
#include "grenade.h"
#include <list>

namespace WeWantWar {


/** @class  BulletTable
 * @brief   Manages and updates the bullets and grenades.
 * @author  Tomi Lamminsaari
 *
 *
 */
class BulletTable
{
public:

  ///
  /// Static methods
  /// ==============
  
  /** Creates new bullet based on given parameters. The bullet will be assigned
   * to given GameObject. The direction the bullet flies will be the same as
   * the directionangle of the given object. If nullpointer was provided as
   * a GameObject, the bullet will move directly up.
   *
   * The bullet type defines the parametervalues we set for the bullet. Those
   * values are requested from @c Settings - class that loads them from
   * general configuration file.
   * @param     aShooter          Pointer to GameObject who shot this.
   * @param     aPos              Starting position of the bullet.
   * @param     aType             Type of the bullet we should make.
   * @return    Pointer to new Bullet.
   */
  static Bullet* createBullet( GameObject* aShooter,
                               const eng2d::Vec2D& aPos,
                               Bullet::TType aType );
                               
  /** Creates new Grenade based on the given parameters. The moving direction
   * of the grenade will be the same where the GameObject is looking at. If
   * no GameObject was given, then the grenade moves directly up.
   * @param     aThrownBy         Pointer to GameObject who threw this.
   * @param     aPos              Starting position of the grenade
   * @param     aType             Type of the Grenade.
   * @return    Pointer to new Grenade.
   */
  static Grenade* createGrenade( GameObject* aThrownBy,
                                 const eng2d::Vec2D& aPos,
                                 Grenade::TType aType );
 
  ///
  /// Public members
  /// ==============

  /** Declare a datatype for iterator being used with this BulletTable. */
  typedef std::list<Bullet*>::iterator BulletIter;
  
  /** Declare a datatype for iterator being used with Grenade - objects in this
   * table.
   */
  typedef std::list<Grenade*>::iterator GrenadeIter;
  

  /** The list where the bullets are being stored. */
  std::list<Bullet*> iBullets;
  
  /** A list for grenades */
  std::list<Grenade*> iGrenades;
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	BulletTable();


	/** Destructor
   */
	virtual ~BulletTable();

private:

	/** Copy constructor.
   * @param     rO                Reference to another BulletTable
   */
  BulletTable( const BulletTable& rO );

	/** Assignment operator
   * @param     rO                Reference to another BulletTable
   * @return    Reference to us.
   */
  BulletTable& operator = ( const BulletTable& rO );

public:


  ///
  /// Methods
  /// =======

  /** Adds the given bullet to this bullet table.
   * @param     aBullet           Pointer to bullet being added. This should
   *                              be dynamically created since it gets deleted
   *                              when we've finished with it.
   */
  void spawnBullet( Bullet* aBullet );
  
  /** Adds the given grenade to the grenade table.
   * @param     aGrenade          Pointer to Grenade being added. This should
   *                              be dynamically created since we delete it ones
   *                              we've done with it.
   */
  void spawnGrenade( Grenade* aGrenade );
  
  /** Clears the contents of the bullet table.
   */
  void cleanup();
  
  /** Updates the bullets in this table.
   */
  void update();
  
  /** Redraws the contents of this bullet table.
   * @param     aTarget           Pointer to bitmap we draw to.
   */
  void redraw( BITMAP* aTarget );
  
  /** Deletes the bullet pointed by the given iterator
   * @param     it                The iterator that defines the position of the
   *                              Bullet we should delete.
   * @return    An iterator to the next Bullet in the table.
   */
  BulletIter deleteBullet( BulletIter it );

  /** Deletes the grenade pointed by the given iterator
   * @param     it                The iterator that defines the position of the
   *                              Grenade we should delete.
   * @return    An iterator to the next Grenade in the table.
   */
  GrenadeIter deleteGrenade( GrenadeIter it );
  
  /** Detonates the grenades thrown by given GameObject.
   * @param     aOwner            Pointer to GameObject whose grenades
   *                              we should detonate.
   */
  void detonateGrenades( GameObject* aOwner );

  

  ///
  /// Getter methods
  /// ==============

  /** Returns the number of bullet there are.
   * @return    Number of bullets.
   */
  int bulletCount() const;
  
  /** Returns the number of Grenades there are.
   * @return    Number of grenades
   */
  int grenadeCount() const;
  
  /** Returns the std - iterator to the first bullet in the bullet table.
   * @return    An iterator to first bullet in the BulletTable
   */
  BulletIter bulletBegin();
  
  /** Returns the std - iterator to first grenade in the bullet table.
   * @return    An iterator to first grenade
   */
  GrenadeIter grenadeBegin();
  
  /** Returns the std - iterator to the last bullet in the bullet table.
   * @return    An iterator to last bullet in the BulletTable
   */
  BulletIter bulletEnd();

  /** Returns the std - iterator to last grenade in the bullet table.
   * @return    An iterator to last grenade
   */
  GrenadeIter grenadeEnd();
  


protected:

  ///
  /// Members
  /// =======
  
  

private:

  ///
  /// Private members
  /// ===============

};


};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: bullettable.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:55:36+02  lamminsa
 * Some methods were removed since bullet class hierarchy was changed.
 *
 */
 
